#!/usr/bin/perl
use Getopt::Std;

## Acquisition des options ------------------------------------------------------
our ($opt_d, $opt_c, $opt_r, $opt_t, $opt_y);
getopts("dcrty:");

## Debug ------------------------------------------------------------------------
our $DEBUG = $opt_d;

## Configuration ----------------------------------------------------------------
our $CFG    = {};
our $STYCFG = "";

## Segments ---------------------------------------------------------------------
our $SEG = [];
our %VAR = ();

## Fichier à traiter ------------------------------------------------------------
our $PREFIXE    = "";
our $REPERTOIRE = "";
our $EXTENSION  = "";
our $SRC        = "";

## Segments de code -------------------------------------------------------------

our @GR = qw(alpha beta gamma delta epsilon zeta eta theta iota kappa lambda mu
   nu xi omicron pi rho sigma tau upsilon phi chi psi omega Alpha Beta Gamma Delta
   Epsilon Zeta Eta Theta Iota Kappa Lambda Mu Nu Xi Omicron Pi Rho Sigma Tau
   Upsilon Phi Chi Psi Omega
);

our %GRT; foreach (@GR) { $GRT{$_} = $_; }

$GRT{theta} = "vartheta";


sub alg_scan {
	my $s = shift;
	unless ($s =~ /[\(\)\[\]\,\-\+\/\*\^\=]/) {
		return $s;
	}
	my $np  =  0; my $nc  =  0; my $tmp = "";
	my $r   = [];
	foreach my $c (split(//, $s)) {
		if ($c eq "(") {
			unless ($np) {
				$tmp and push @$r, "($tmp)";
				$tmp  = "";
			} else {
				$tmp .= "(";
			}
			$np++;
		} elsif ($c eq ")") {
			$np--;
			unless ($np) {
				$tmp and push @$r, &alg_scan($tmp);
				$tmp  = "";
			} else {
				$tmp .= ")";
			}
		} elsif ($c eq "[") {
			unless ($nc) {
				$tmp and push @$r, "[$tmp]";
				$tmp = "";
			} else {
				$tmp .= "[";
			}
			$nc++;
		} elsif ($c eq "]") {
			$nc--;
			unless ($nc) {
				$tmp and push @$r, &alg_scan($tmp);
				$tmp = "";
			} else {
				$tmp .= "]";
			}
		} elsif (not($np) and not($nc) and $c =~ /[\,\-\+\/\*\^\=]/) {
			$tmp and push @$r, &alg_scan($tmp);
			push @$r, $c; $tmp = "";
		} else {
			$tmp .= $c;
		}
	}
	$tmp and push @$r, $tmp;
	return $r;
}

sub alg_est_atome {
	my $s = shift;
	$s =~ /[\^\+\-\;\(]/ and return 0;
	return 1;
}

sub alg_tex {
	my $r = shift;
	if (ref $r) {
		# Table des fonctions, retenues pour simplifications ultérieures.
		my $fonc = []; my $ifonc = 0; # $ifonc est un index
		# Réduction des fonctions, des groupes, des variables indicées et des
		# variables transformables.
		my $i = 0; foreach (@$r) {
			if ($_ =~ /^\((.+)\)$/) {
				$$r[$i] = "<<$ifonc>>";
				$$fonc[$ifonc++] = {nom => $1, arg => &alg_tex($$r[$i+1]) };
				$$r[$i+1] = "";
			} elsif ($_ =~ /^\[(.+)\]$/) {
				$$r[$i] = "{\\mathrm{$1}_{" . &alg_tex($$r[$i+1]) . "}}";
				$$r[$i+1] = "";
			} elsif (ref $_) {
				$$r[$i] = "{" . &alg_tex($_) . "}";
			} elsif (exists $GRT{$_}) {
				$$r[$i] = "\\$GRT{$_}";
			}
			$i++;
		}
		# Reconstruction de la table
		my $rr = []; foreach (@$r) { $_ and push @$rr, $_; }
		# Reduction des exposants
		$i = 0; foreach (@$rr) {
			if ($_ eq "^") {
				my $a = $$rr[$i-1];
				if ($a =~ /<<(\d+)>>/) {
					my $nom = $$fonc[$1]->{nom};
					my $arg = $$fonc[$1]->{arg};
					alg_est_atome($arg) or $arg = "\\left($arg\\right)";
					$$rr[$i] = "\\mathrm{$nom}^" . $$rr[$i+1] . $arg;
				} else {
					alg_est_atome($a) or $a = "\\left($a\\right)";
					$$rr[$i] = $a . "^" . $$rr[$i+1];
				}
				$$rr[$i-1] = $$rr[$i+1] = "";
			}
			$i++;
		}
		# Reconstruction de la table
		$r = []; foreach (@$rr) {$_ and push @$r, $_; }
		# Réduction des divisions, des fonctions et transformation de "*"
		$i = 0; foreach (@$r) {
			if ($_ eq "/") {
				my $n = $$r[$i-1];
				my $d = $$r[$i+1];
				$$r[$i] = "\\frac{$n}{$d}";
				$$r[$i-1] = $$r[$i+1] = "";
			} elsif (/<<(\d+)>>/) {
				my $f = $$fonc[$1];
				alg_est_atome($$f{arg}) or $$f{arg} = "\\left($$f{arg}\\right)";
				$$r[$i] = "\\mathrm{$$f{nom}}\\,$$f{arg}\\,";
			} elsif ($_ eq "*") {
				$$r[$i] = "\\;";
			}
			$i++;
		}
		return join("", @$r);
	} else {
		exists $GRT{$r} and return "\\$GRT{$r}";
		return $r;
	}
}




sub ConstruitComplementTeX {
	my $s = shift;
	my @l = split(/\s*\n/, $s);
	foreach my $e (@l) {
		# Extraction des chaînes (%t5) (exemple) précédant l'espression
		# numérique.
		$p = ""; if ($e =~ s/^\s*\(%([a-z]+)(\d+)\)\s+//) {
			$p = "(\\%$1$2)\\;\\; ";
		}
		$e =~ s/ //g;
		$e = "\\($p" . alg_tex(alg_scan($e)) . "\\)";
	}
	my $tex = join("\\\\[1mm]\n", @l);
	return EvalConfiguration($CFG,"COMPTEX",$STYCFG,["", $tex]);
};
#S === Procédures d'appui ------------------------------------------------------

# Obtenir le contenu d'un fichier texte sous la forme d'une référence à une table
# des lignes.
sub ReadFichierRef {
	my $f = shift;
	open(FICH, $f) or die "Le fichier $f est introuvable !\n";
	my @L = <FICH>;
	close FICH;
	chomp @L;
	return \@L;
}

# Obtenir le contenu d'un fichier texte en une seule chaîne.
sub ReadFichierString {
	my $f = shift;
	local $/;
	open(FICH, $f) or die "Le fichier $f est introuvable !\n";
	my $c = <FICH>;
	close FICH;
	return $c;
}

# Obtenir le préfixe, le répertoire et l'extension (sans le point) d'un fichier.
sub GetFichierPRE {
	my $f = shift;
	use File::Basename;
	my ($n, $r, $e) = fileparse($f,qw{\..*});
	$e =~ s/^\.//;
	return ($n, $r, $e);
}

# Écrire une table de données dans un fichier yaml.
sub YamlWriteFichier {
	my ($d, $f) = @_;
	use YAML::Tiny;
	my $y = YAML::Tiny->new();
	my $i = 0;
	foreach (@$d) {
		$y->[$i++] = $_;
	}
	$y->write("$f.yml");
}

# Obtenir une table de données contenues dans un fichier yaml.
sub YamlReadFichier {
	my $f = shift;
	use YAML::Tiny;
	my $y = YAML::Tiny->read("$f.yml");
	my $d = [];
	my $i = 0;
	foreach (@$y) {
		$d->[$i++] = $_;
	}
	return $d;
}

# Obtenir une configuration à partir du segment DATA d'un script ou d'un
# fichier.
sub GetConfiguration {
	my ($d, $f) = @_;
	my @s = ();
	local $/;
	unless ($f) {
		@s = split /^\[(.+)\]\s*\n/m, <DATA>; shift @s;
	} else {
		open (DTA, "$f.cfg") or die "Le fichier de configuration $f est introuvable!\n";
		@s = split /^\[(.+)\]\s*\n/m, <DTA>; shift @s;
		close DTA;
	}
	while (@s) {
		my ($nom, $contenu) = splice(@s, 0, 2);
		$contenu =~ s/^\s*|\s*$//g;
		$d->{$nom} = $contenu;
	}
	return 1;
}

# Écrire une configuration dans un fichier ou sur la sortie standard.
sub WriteConfiguration {
	my ($d, $f) = @_;
	unless ($f) {
		foreach (sort keys %$d) {
			my $s = $$d{$_};
			$s =~ s/\s*$//;
			print  STDOUT "[$_]\n" . $s . "\n\n";
		}
	} else {
		open CFG, ">$f.cfg";
		foreach (sort keys %$d) {
			my $s = $$d{$_};
			$s =~ s/\s*$//;
			print  CFG "[$_]\n" . $s . "\n\n";
		}
		close  CFG;
		print STDERR "Le fichier de configuration ($f.cfg) est écrit...\n";
	}
	return 1;
}

# Évaluer un élément de configuration.
sub EvalConfiguration {
	my ($c, $k, $s, $v) = @_;
	my $p = exists $$c{"$s-$k"} ? $$c{"$s-$k"} : $$c{$k};
	$p =~ s/\$(\d)/"defined \$\$v[$1] ? \$\$v[$1] : ''"/gee;
	$p =~ s/\&nl/\n/g;
	return $p;
}

# Interpolation : bloc SI
sub SI {
	my ($r, $c, $t) = @_;
	return $t if $$r{$c};
	return "";
}

# Interpolation : bloc SINON
sub SINON {
	my ($r, $c, $t) = @_;
	return $t unless $$r{$c};
	return "";
}

# Interpolation d'un segment de texte.
sub Interpolation {
	my ($p, $r) = @_;
	$p =~ s/S\!([a-zA-Z_0-9]+):?(.*?)\!/"defined \$r->{$1} ? \$r->{$1} : '$2'"/gee; # Substitution
	$p =~ s/SI\!([a-zA-Z_0-9]+):(.*?)\!FINSI/&SI($r,$1,$2)/sge;                     # Si ...
	$p =~ s/SINON\!([a-zA-Z_0-9]+):(.*?)\!FINSI/&SINON($r,$1,$2)/sge;               # Sin non ...
	return $p;
}

# Conversion d'un fichier eps en un fichier pdf.
sub EPS2PDF {
	my $f = shift;
	# Commande pour déterminer la BoundingBox d'un fichier EPS via GS.
	my $GSBBOX = "gs -q -sDEVICE=bbox -sPAPERSIZE=a4 -dNOPAUSE -dNOSAFER";
	my @b = split /\s+/, qx{$GSBBOX $f.eps quit.ps 2>&1 | grep "%%BoundingBox"};
	shift @b;
	my ($l,$h) = ($b[2] - $b[0], $b[3] - $b[1]); 
	open(PS,">temp-$f.ps");
	print PS "<</PageSize [$l $h] >> setpagedevice\n";
	print PS "$b[0] neg $b[1] neg translate ($f.eps) run\n";
	close(PS);
	qx{ps2pdf -dNOSAFER temp-$f.ps $f.pdf };
	unlink "temp-$f.ps" if -f "temp-$f.ps";
}

# Conversion d'un fichier eps en un fichier png.
sub EPS2PNG {
	my $f = shift;
	qx{convert $f.eps $f.png};
}


# ==============================================================================
# === Serialisation ------------------------------------------------------------
# ==============================================================================
sub SerialisationLectureOptions {
	# Lecture des options placées sur la ligne de déclaration d'un bloc ---
	# TODO : À développer !
	my ($l, $o) = @_;
	$l =~ s/^\s*|\s*$//g;
	$l or return;
	my @s = split /\[(.+?)\]\s+/m, " $l"; shift @s;
	while (@s) {
		my ($nom, $contenu) = splice(@s, 0, 2);
		$contenu =~ s/^\s*|\s*$//g;
		$contenu =~ s/\&nl/\n/g;
		if ($nom =~ s/^\*//) {
			$$CFG{$nom} = $contenu;
		} else {
			$$o{$nom} = $contenu;
		}
	}
}

sub SerialisationEnregistrement {
	my ($r, $t, $c, $p, $o, $n) = @_;
	# Test sur l'existence du segment
	ref $r->[$n] or $r->[$n] = {};
	# Suppression des blancs, éventuels, précédant la commande
	$c  =~ s/^\s*//;
	my $d = "";
	$c  =~ s/^\*// and $d = "*";
	$c  =~ s/^\!// and $d = "!";
	my $f = "";
	$c  =~ s/\-$//   and $f = "-";
	$c  =~ s/\+$//   and $f = "+";
	$c  =~ s/\*\*$// and $f = "**";
	$c  =~ s/\*$//   and $f = "*";
	# Affectations.
	$r->[$n]->{txt} = $t;    # texte
	$r->[$n]->{cmd} = $c;    # commande
	$r->[$n]->{pre} = $d;    # avant la commande
	$r->[$n]->{pst} = $f;    # après la commande
	$r->[$n]->{pls} = $p;    # plus
	$r->[$n]->{opt} = $o;    # options
}
sub Serialisation {
	# Référence à la table de lignes
	my $l    = shift;
	# Référence à la table des blocs
	my $r    = shift;
	# Compteur des blocs reconnus
	my $n    = 0;
	# Drapeau de bloc commande
	my $flg  = 0;
	# Drapeau des ciseaux...
	my $cut  = 0;
	my $txt  = "";
	my $cmd  = "";
	my $pls  = {};
	my $src  = "";
	my $opt  = {};
	foreach (@$l) {
		# Arrêt de la lecture si la ligne commence par ...
		last if /^\.\.\./;
		# Inscription dans le source effectif.
		$src .= "$_\n";
		# Partie non soumise au traitement
		if (/^8</) {
			$cut = 1 - $cut;
			next;
		}
		if ($cut) {
			$txt .= "$_\n";
			next;
		}
		my $s = $_;
		# C'est une commande !
		if (s/^\.\s+(.)/$1/) {
			s/\s+$//;
			if ($flg eq "TIRET") {
				$cmd .= "\n$s";
				next;
			}
			# Entrée dans le mode « point »
			$opt->{mode} = "POINT";
			# Clôture du bloc texte.
			$txt  =~ s/\s*$//;
			$txt .= "\n";
			if ($flg) {
				# Enregistrement
				&SerialisationEnregistrement($r, $txt, $cmd, $pls, $opt, $n++);
				$txt = $cmd = ""; $pls = {}; $opt = {};
			}
			# C'est le début d'un bloc commande.
			$cmd  = $_;
			$flg = "POINT";
			# Inutile d'en faire plus...
			next;
		}
		if (s/^\-\-\-\s*//) {
			# Entrée dans le mode « tiret »
			$opt->{mode} = "TIRET";
			&SerialisationLectureOptions($_, $opt);
			# C'est le début d'un bloc commande
			$flg = "TIRET";
			# Inutile d'en faire plus
			next;
		}
		if ($flg) {
			# On est dans un bloc commande... (ci-dessous : conditions de terminaison)
			if ($_ and $_ !~ /^\s*\\/ and $_ !~ /^\./ and $_ !~ /^%/) {
				# Nous y restons...
				$cmd .= "\n$_";
				# Inutile d'en faire plus...
				next;
			} else {
				# Le bloc commande est terminé, clôture et enregistrement.
				$txt  =~ s/\s*$//;
				$txt .= "\n";
				&SerialisationEnregistrement($r, $txt, $cmd, $pls, $opt, $n++);
				$txt = $cmd = ""; $pls = {}; $opt = {};
				$flg = 0;
			}
		}
		# Élément de configuration : les plus..
		if (/^\.\+\s*(.+)/) {
			&SerialisationLectureOptions($_, $pls);
			# Inutile d'en faire plus...
			next;
		}
		$txt .= "$_\n";
	}
	if ($txt or $cmd) {
		$txt  =~ s/\s*$//;
		$txt .= "\n";
		&SerialisationEnregistrement($r, $txt, $cmd, $pls, $opt, $n++);
	}
	return \$src;
}



sub Prompt ($;$) {
	my($mess, $def) = @_;
	Carp::confess("Appel de Prompt sans arguments...")
		unless defined $mess;
	my $isa_tty = -t STDIN && (-t STDOUT || !(-f STDOUT || -c STDOUT)) ;
	my $dispdef = defined $def ? "[$def] " : " ";
	$def = defined $def ? $def : "";
	local $|=1;
	local $\;
	print "$mess $dispdef";
	my $ans;
	if ($ENV{PERL_MM_USE_DEFAULT} || (!$isa_tty && eof STDIN)) {
		print "$def\n";
	} else {
		$ans = <STDIN>;
		if( defined $ans ) {
			chomp $ans;
		} else { # user hit ctrl-D
			print "\n";
		}
	}
	return (!defined $ans || $ans eq '') ? $def : $ans;
}



### --- Variables globales propres à la session --------------------------------
our $SMAXFIG = 0;  # Figures crees
### ----------------------------------------------------------------------------
sub MaximaPlotFigure {
	$SMAXFIG++;
	return sprintf("$$CFG{DIRFIG}$PREFIXE%02d", $SMAXFIG);
}
### ----------------------------------------------------------------------------
sub MaximaPlotX {
	my ($r, $s) = @_;
	# Extraction de la commande pour xmaxima. Il y aura certainement à faire
	# les choses moins brutalement.
	# La chaîne $s pourrait contenir autre chose à la suite.
	$s =~ s/^\{|\}$//g;
	open(TMP, ">maxplot_temp"); print TMP $s; close(TMP);
	open(CFG,">.printOptions"); print CFG $$CFG{MAXPLOTOPTIONS}."\n"; close(CFG);
	# Réaffectation temporire de la variable d'environnement HOME utilisée par
	# xmaxima pour localiser les options d'impression. Puis exécution de xmaxima.
	qx{export HOME=.;xmaxima maxplot_temp};
	if (-f "maxplot.eps") {
		my $fig = MaximaPlotFigure();
		rename("maxplot.eps","$fig.eps");
		$$r{fig} = $fig;
	}
	# Suppression du fichier d'appel temporaire.
	$DEBUG or unlink("maxplot_temp");
	# Suppression du fichier de configuration de l'impression.
	unlink(".printOptions");
	# On retourne une chaîne vide.
	return "";
}
### ----------------------------------------------------------------------------
# Récupération du fichier data.gnuplot qui est écrit dans le HOME.
# Cette procédure risque de poser des problèmes de synchronisation. 
# Est-ce qu'il s'agira du bon fichier ? Il peut être ancien...
# On ne peut se permettre de le supprimer une fois copié, il est 
# peut-être en cours d'utilisation ! De plus son écriture n'est pas
# nécessairement achevée...
sub MaximaPlotOutDataGnuplot {
	use File::stat;
	my $f = shift; my $dta = "$ENV{HOME}/data.gnuplot";
	if (-f $dta) {
		my $n = 0; my $j = -1;
		while ($n < 5) {
			my $s = stat($dta);
			$n = $s->size == $j ? $n + 1 : 0;
			$j = $s->size;
		}
		copy("$dta","$f.dta");
	return 1;
	}
	return 0;
}
### ----------------------------------------------------------------------------
sub MaximaPlotOutGnuplot {
	use File::Copy;
	my $f = shift;
	if (-f "$ENV{HOME}/maxout.gnuplot") {
		copy("$ENV{HOME}/maxout.gnuplot","$f.gp");
		# Adaptation du chemin d'accès aux données. On commence par
		# déterminer le prefixe des figures (qui contient éventuellement
		# l'indication d'un chemin).
		my ($n, $r, $e) = GetFichierPRE($f);
		qx{sed -i 's#$ENV{HOME}/data\.gnuplot#$n.dta#g' $f.gp};
		qx{sed -i 's/maxima_out\.eps/$n.eps/' $f.gp};
	}
}
### ----------------------------------------------------------------------------
sub MaximaPlotOutEps {
	my $r = shift;
	if (-f "maxima_out.eps") {
		my $fig = MaximaPlotFigure();
		MaximaPlotOutGnuplot($fig) if MaximaPlotOutDataGnuplot($fig);
		rename("maxima_out.eps","$fig.eps");
		$$r{fig} = $fig;
	} else {
		my $answer = Prompt($$CFG{_PROMPTOUTEPS},"O");
		&MaximaPlotOutEps($e) if $answer eq "O";
	}
}




### --- Adaptation de la commande : production de figure ou sortie latex -------
sub SessionEntree {
	my ($r, $rang) = @_;
	my $cmd = $$r{cmd};
	# La terminaison est fixée à ; si absente. Cela évite le blocage du serveur.
	$cmd =~ s/(\;|\$)\s*$//s; my $t = $1; $t = ";" unless $t;
	if ($cmd =~ /^plot([23])d\((.*)\)$/s) {
		my $fig = MaximaPlotFigure();
		my $cmdm = EvalConfiguration($CFG,"PLOTGNUEPS",$STYCFG,["plot$1d","$2","$fig.eps"]);
		return { rang => $rang, maxima => "$cmdm$t", commande => "$cmd$t", figure => $fig };
	} elsif ($$r{pre} eq "*") {
		my $fig = MaximaPlotFigure();
		$cmd =~ /([^\(]+)\((.+)\)/s;
		my $cmdm = EvalConfiguration($CFG,"PLOTGNUEPS",$STYCFG,["$1","$2","$fig.eps"]);
		$DEBUG and print STDERR "pmaxima> * $cmd --> $cmdm\n";
		return { rang => $rang, maxima => "$cmdm$t", commande => "$cmd$t", figure => $fig, gnuplot => 1};
	} elsif ($cmd =~ /^qdraw\s*\((.*)\)$/s) {
		my $fig = MaximaPlotFigure();
		my $cmdm = EvalConfiguration($CFG,"QDRAWGNUEPS",$STYCFG,["$1",$fig]);
		return { rang => $rang, maxima => "$cmdm$t", commande => "$cmd$t", figure => $fig };
	} else {
		return { rang => $rang, maxima => "tex($cmd)$t", commande => "$cmd$t"};
	}
}
sub SessionSortiePMX {
	my $r = shift;
	if ($$r{comp} =~ s/<pmx>(.*)<\/pmx>//) {
		# print STDERR "$1\n";
		my $pmx = $1;
		if ($pmx =~ /<gr>(.*)<\/gr>/) {
			# print STDERR "$1\n";
			return EvalConfiguration($CFG,"FIGURE",$STYCFG,[$$CFG{VARFIGURE},$1]);
		}
	}
	return "";
}
### --- Enregistrement de la sortie --------------------------------------------
sub SessionSortie {
	my ($e, $s) = @_;
	# Référence pour l'enregistrement des champs a conserver
	my $r = {};
	exists $$e{figure} and $$r{fig} = $$e{figure};
	# Extraction de la dernière ligne éventuelle ((%oxx)...)
	if ($s =~ s/\(%o(\d+)\)(.*)$//s) {
		$$r{id} = $1; $$r{o} = $2;
	}
	# Suppression d'un bloc LaTeX vide...
	$s =~ s/\$\$\$\$//;
	# Extraction du contenu latex éventuel.
	$$r{ltx} = $s =~ s/\$\$(.+?)\$\$//s ? $1 : "";
	# Extraction du verbatim éventuel.
	if ($s =~ s/\\begin\{verbatim\}(.*?)\\end\{verbatim\}//s) {
		$$r{verb} = $1;
	}
	# Nettoyage de ce qui reste.
	$s =~ s/^\s*|\s*$//g;
	$s =~ /^\{plotdf/ and $s = MaximaPlotX($r, $s);
	# $s =~ /^\{plot2d/ and $s = MaximaPlotX($r, $s);
	$$r{comp} = $s;
	# Recherche d'une sortie spéciale <pmx>...</pmx>
	$$r{pmx} = SessionSortiePMX($r);
	# Récupération d'une image dans le cas de l'utilisation de draw[23]d, draw...
	$$e{commande} =~ /^draw[23]d\s*\(/ and MaximaPlotOutEps($r);
	$$e{commande} =~ /^draw\s*\(/      and MaximaPlotOutEps($r);
	# Récupération d'un fichier gnuplot à la racine du HOME
	$$e{gnuplot}
		and rename("$ENV{HOME}/maxout.gnuplot","$$e{figure}.gnu")
		and qx{rm -f $$e{figure}.gnu.gz && gzip $$e{figure}.gnu};
	return $r;
}
### --- Ventilation des entrees/sorties ----------------------------------------
sub SessionEntreeSortie {
	my ($e, $s, $id) = @_;
	# Enregistrement de la sortie precedente
	my $r = SessionSortie($e, $s);
	my $n = $$e{rang};
	if ($n == -1) {
		# Acquisition de la banière présentant la version de maxima utilisée.
		my $b = $$r{comp};
		if ($b =~ /pid/ and -f "./pmaxima.intro") {
			my $bl = ReadFichierRef("./pmaxima.intro");
			shift @$bl;
			$b = join("\n", @$bl[0,1,2,3]);
			$$CFG{BANNIERE} = EvalConfiguration($CFG,"BANNER","",[$b]);
			$DEBUG or qx{rm -f ./pmaxima.intro};
		}
		$DEBUG and print STDERR "pmaxima> banner:\n$b\n";
	} else {
		$$SEG[$n]->{out} = $r;
	}
	$n++;
	$$SEG[$n]->{cmd} and return SessionEntree($$SEG[$n],$n);
	return "";
}
### --- Session Maxima ---------------------------------------------------------
sub Session {
	use POSIX;
	use IO::Socket;
	# Le port utilisé pour communiquer -------------------------------------------
	my $port    = 9001;
	# Création d'une prise -------------------------------------------------------
	my $serveur = IO::Socket::INET->new( Proto     => 'tcp',
                                             LocalPort => $port,
                                             Listen    => SOMAXCONN,
                                             Reuse     => 1);
	# En cas d'erreur ------------------------------------------------------------
	$serveur or die "$$CFG{_ERREURSERVEUR}\n";
	# Le père est bien installé, on crée le fils  --------------------------------
	my $pidp = $$;
	my $pidf = fork();
	# Séparation des tâches du fils et du père -----------------------------------
	unless ($pidf) {
		# Le fils lance maxima...
		sleep 0.5;
		my $cmd = EvalConfiguration($CFG,"MAXIMA","",[$port]);
		$cmd .= " 1> ./pmaxima.intro 2>&1";
		# system("maxima -s $port >/dev/null");
		system($cmd);
		$DEBUG and print STDERR "pmaxima> Le client est arrêté !\n";
		exit(1);
	} else {
		$DEBUG and print STDERR "pmaxima> Le serveur est actif - père : $pidp, fils : $pidf\n";
		# Création de la ligne à exécuter pour en finir avec le script, en cas de 
		# besoin.
		# qx{echo "kill -9 $PIDP $PIDF" > pid};
		# Le père est à l'écoute du fils...
		my $id     = 0;
		my $entree = { rang => -1, maxima => "", commande => "Lancement de Maxima..."};
		my $sortie = "";
		my $tampon = "";
		while (my $client = $serveur->accept()) {
			$client->autoflush(1);
			while (1) {
				sysread($client, $tampon, POSIX::BUFSIZ);
				$sortie .= $tampon;
				if ($sortie =~ s/\(%i([0-9]+)\) $//) {
					# Maxima est en attente (%i$1)...
					# Récupération du numéro d'ordre de l'instruction
					# qui va être soumise à MAXIMA.
					$id = $1;
					# Traitement de la dernière sortie de MAXIMA et
					# récupération de la nouvelle entrée.
					$entree = SessionEntreeSortie($entree, $sortie, $id-1);
					if (ref $entree) {
						# Une entrée est proposée.
						$sortie = "";
						$DEBUG and print STDERR "pmaxima> $$entree{maxima}\n";
						print $client "$$entree{maxima}\n";
					} else {
						# C'est la fin du traitement, on termine la session.
						print $client "quit();\n";
						last;
					}
				}
			}
			close $client;
			last;
		}
	}
	# Suppression de la prise ----------------------------------------------------
	close $serveur;
	$DEBUG and print STDERR "pmaxima> le serveur est arrêté !\n";
}


## -----------------------------------------------------------------------------
sub InclusionResultatTeX{
	my ($l,$v) = @_;
	exists $VAR{$v} and return $VAR{$v}->{ltx};
	return $l;
}
## -----------------------------------------------------------------------------
sub EnregistrementTeXVar {
	my $c = shift; my $v = ""; my $d = "";
	if ($c =~ /^(.+?)\s*:\s*(.+)/) {
		$v = $1; $d = $2;
	}
	return $v;
}
## -----------------------------------------------------------------------------
sub EnregistrementTeXCommande {
	my ($t, $i) = @_;
	chomp($t);
	$t =~ s/\\\n/\n/g;
	if ($$CFG{ALLTT}) {
		$t =~ s/\\/\\(\\backslash\\)/g;
		$t =~ s/\-\>/\\(\\rightarrow\\)/g;
		$t =~ s/\<\=/\\(\\le\\)/g;
		$t =~ s/\>\=/\\(\\ge\\)/g;
		$t =~ s/\:\=/\\(:=\\)/g;
		$t =~ s/\:\>/\\(:>\\)/g;
		$t =~ s/\{/\\{/g;
		$t =~ s/\}/\\}/g;
		# $t =~ s/(.+?)_(.+?)\^(.+?)/\\($1\\sb\{$2\}\\sp\{$3\}\\)/g;
		# $t =~ s/(.+?)_(.+?)/\\($1\\sb\{$2\}\\)/g;
		$t =~ s/\(\*(.+?)\*\)/\\text\{\\color\{LightSlateGrey\}(*$1*)\}/g;
		# Échappement de certains caractères : { et } dans les commentaires par exemple.
		$t =~ s/\#\\(.)/$1/g;
	}
	return  &EvalConfiguration($CFG,"COMMAND",$STYCFG,[$t,$i+1]);
}
## Enregistrement de la partie complémentaire ----------------------------------
sub EnregistrementTeXComp {
	my ($s, $i) = @_;
	my $j = sprintf("%03d",$i);
	open  COMP, "> ${PREFIXE}-C$j.txt";
	print COMP $$s{out}->{comp};
	close COMP;
}
## Traitement des .plus ! ------------------------------------------------------
sub EnregistrementTeXPlus {
	my $s = shift;
	foreach (keys %{$$s{pls}}) {
		# (03/06/13) substitution d'un élément de configuration
		if (exists $$CFG{$_}) {
			$$CFG{$_} = $s->{pls}->{$_};
		}
	}
	return "";
}
## -----------------------------------------------------------------------------
sub EnregistrementTeXResultat {
	my ($s, $l) = @_;
	my $r = $$s{out};
	my $out = "";
	if ($$r{ltx} ne "") {
		unless ($$s{pst} eq "*" or $$s{cmd} =~ /\$$/) {
			$out .= EvalConfiguration($CFG,"RESULTAT",$STYCFG, [$$r{id},$$CFG{BMATH}.$$r{ltx}.$$CFG{EMATH}]);
		}
		my $v = EnregistrementTeXVar($$s{cmd});
		$v and $VAR{$v} = $r;
		$$l = $$r{ltx};
	}
	if ($$r{fig}) {
		unless ($opt_r) {
			qx{epstopdf $$r{fig}.eps -o $$r{fig}.pdf};
		}
		$out .= EvalConfiguration($CFG,"FIGURE",$STYCFG, [$$CFG{VARFIGURE}, $$r{fig}]);
	}
	if ($$r{verb} and $$r{ltx}) {
		$$r{verb} =~ s/^\n//;
		my $verbatim = join("|\\\\\n\\verb|", split(/\s*\n/, $$r{verb}));
		$out .= EvalConfiguration($CFG, "VERBATIM", $STYCFG, [$verbatim]);
	}
	if ($$r{comp} and $$s{pst} ne "**") {
		if ($$CFG{COMPTEX}) {
			$out .= ConstruitComplementTeX($$r{comp});
		} else {
			my $verbatim = join("|\\\\\n\\verb|", split(/\s*\n/, $$r{comp}));
			$out .= EvalConfiguration($CFG, "COMPLEMENT", $STYCFG, [$verbatim]);
		}
	}
	if ($$r{pmx}) {
		$out .= $$r{pmx};
	}
	return $out;
}
## -----------------------------------------------------------------------------
sub EnregistrementTeX {
	my $last = "";
	open(TEX,">$PREFIXE.$$CFG{EXTENSION}");
	my $is = scalar @$SEG;
	for(my $i = 0; $i < $is; $i++) {
		# Le segment à traiter...
		my $s   = $$SEG[$i];
		$$s{pls} and print TEX EnregistrementTeXPlus($s);
		$$s{pst} eq "**" and EnregistrementTeXComp($s,$i);
		my $txt = $$s{txt};
		# Éventuelles inclusion d'éléments de configuration...
		$txt  = Interpolation($txt, $CFG);
		# Éventuelles inclusions de résultats
		$txt  =~ s/\.%([\da-zA-Z]*)\./InclusionResultatTeX($last, $1)/ge;
		# Éventuelles évaluations... (expérimental et simpliste !)
		$txt  =~ s/EVAL\((.+?)\)/eval($1)/ge;
		# ----- Fin des transformations de la partie texte
		print TEX $txt;
		$last = $s->{out}->{ltx};
		my $cmd = $$s{cmd};
		$$s{pst} eq "-" and next;
		if ($cmd) {
			print TEX EnregistrementTeXCommande($cmd, $i);
		}
		print TEX EnregistrementTeXResultat($s, \$last);
	}
	close(TEX);
}


sub ExtractionYml {
	my $t = shift;
	-f "$PREFIXE.yml" or die "Le fichier requis $PREFIXE.yml est absent...\n";
	my $r = YamlReadFichier($PREFIXE);
	my $SEG = $$r[0];
	open OUT, ">$PREFIXE-$t.out";
	foreach (@$SEG) {
		print OUT "$$_{$t}\n";
	}
	close OUT;
	return 1;
}
## =========================================================================== ##
##                               TRAITEMENT                                    ##
## =========================================================================== ##
if ($ARGV[0]) {
	-f $ARGV[0] or die "Le fichier $ARGV[0] est absent...\n";
	($PREFIXE,$REPERTOIRE,$EXTENSION) = GetFichierPRE($ARGV[0]);
	## Extraction du fichier YML --------------------------------------------
	$opt_y and ExtractionYml($opt_y) and exit(0);
}
## Lecture de la section DATA et configuration ----------------------------------
GetConfiguration($CFG);
$opt_c and WriteConfiguration($CFG,"pmaxima") and exit(0);
GetConfiguration($CFG,"$ENV{HOME}/.wanda/pmaxima") if -f "$ENV{HOME}/.wanda/pmaxima.cfg";
GetConfiguration($CFG,"../pmaxima") if -f "../pmaxima.cfg";
GetConfiguration($CFG,"pmaxima") if -f "pmaxima.cfg";
## === Session ------------------------------------------------------------------
if ($opt_r) {
	-f "$PREFIXE.yml" or die "Le fichier requis $PREFIXE.yml est absent...\n";
	my $r = YamlReadFichier($PREFIXE);
	$SEG = $$r[0];
	$$CFG{BANNIERE} = $$r[2];
	my $s = Serialisation(ReadFichierRef($ARGV[0]),$SEG);
	$SRC = $$s;
} else {
	my $s = Serialisation(ReadFichierRef($ARGV[0]),$SEG);
	$SRC = $$s;
	Session();
}
## == Finalisation --------------------------------------------------------------
&EnregistrementTeX();
&YamlWriteFichier([$SEG,$SRC,$$CFG{BANNIERE}],$PREFIXE);
if ($opt_t) {
	my $cmd = $$CFG{TEX}; chomp($cmd);
	qx{$cmd $PREFIXE.$$CFG{EXTENSION}};
}


1;


__DATA__

[COMPTEX]
0

[ALLTT]
1

[MAXIMA]
/usr/lib/maxima/5.27.0/binary-gcl/maxima -eval '(cl-user::run)' -f '--' -r ':lisp (start-client $0)'


[EXTENSION]
tex

[_ERREURSERVEUR]
Le serveur ne peut être lancé !

[_PROMPTOUTEPS]
Le fichier maxima_out.eps n'est pas prêt, attendre (0,N) ?

[DIRFIG]
./

[TEX]
w2tex -c

[COMMAND]
&nl{\color{gray}
\begin{quote}
\begingroup\leavevmode
\llap{\footnotesize\textbf{\itshape m]\hspace{1em}}}%
\begin{minipage}[t]{\linewidth}\small
\begin{alltt}
$0
\end{alltt}
\end{minipage}
\endgroup
\end{quote}
}&nl&nl

[BMATH]
\(

[EMATH]
\)

[RESULTAT]
\begin{quote}
\begingroup\leavevmode
\llap{\footnotesize\color{orange}\%o$0 : }{\small\color{teal}$1}
\endgroup
\end{quote}
&nl&nl

[FIGURE]
\begin{center}
\includegraphics[$0]{$1.pdf}
\end{center}

[VARFIGURE]
width=0.7\linewidth

[COMPLEMENT]
{\small\color{purple}\verb|$0|}
&nl&nl

[VERBATIM]
{\small\color{brown}\verb|$0|}
&nl&nl

[PLOTGNUEPS]
$0($1,[plot_format, gnuplot], [gnuplot_term, ps], [gnuplot_out_file, "$2"])

[QDRAWGNUEPS]
qdraw($0, pic(eps,"$1", font("Arial",9)))

[MAXPLOTOPTIONS]
set printOption(landscape) 0
set printOption(papersize) A4
set printOption(hoffset) 0.5
set printOption(voffset) 0.5
set printOption(xticks) 20 
set printOption(yticks) 20
set printOption(psfilename) "maxplot.eps"
set printOption(centeronpage) 1

[MACROS]
\def\Maxima{{\color[HTML]{881122}\itshape\textbf{Maxima}}}
\def\pmaxima{{\color[HTML]{118822}\itshape\textbf{pmaxima}}}

[BANNER]
{\small\setlength{\baselineskip}{1.1em}
\begin{minipage}{0.11\linewidth}\centering
\includegraphics[width=0.8\linewidth]{maxima-new.png}
\end{minipage}\hfill%
\begin{minipage}[c]{0.85\linewidth}
\begin{verbatim}
$0
\end{verbatim}
\end{minipage}
}\\
\hrule
